import express from "express";
import { chromium, devices } from "playwright";
import fs from "fs";
import path from "path";

const app = express();
const PORT = 3131;
const iPhone = devices["iPhone 13"];

// config.json'dan tüm siteleri oku
const configPath = path.join(process.cwd(), "config.json");
if (!fs.existsSync(configPath)) {
  throw new Error("config.json bulunamadı!");
}

const sites = JSON.parse(fs.readFileSync(configPath, "utf-8"));

async function scrape(siteConfig) {
  const { name, domain, subDomain, divSelector } = siteConfig;
  const url = `${subDomain}.${domain}`;
  
  const browser = await chromium.launch({ headless: true });
  const context = await browser.newContext({
    ...iPhone,
    userAgent:
      "Mozilla/5.0 (iPhone; CPU iPhone OS 15_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/15.0 Mobile/15E148 Safari/604.1",
  });
  const page = await context.newPage();
  
  await page.goto(`https://${url}/tr/`);
  await page.waitForSelector(divSelector, { timeout: 15000 });
  
  let content = await page.locator(divSelector).innerHTML();
  
  // HTML içinde hem www hem de subdomain linklerini replace et
  const regexWww = new RegExp(`https://www\\.${domain.replace(/\./g, '\\.')}/`, "g");
  const regexSubdomain = new RegExp(`https://${subDomain.replace(/\./g, '\\.')}\\.${domain.replace(/\./g, '\\.')}/`, "g");
  
  content = content.replace(regexWww, "/");
  content = content.replace(regexSubdomain, "/");
  
  await browser.close();
  
  const fileName = `matches-${name}.html`;
  fs.writeFileSync(fileName, content, "utf-8");
  
  const meta = { 
    lastFetched: new Date().toISOString(),
    site: name,
    domain: url
  };
  fs.writeFileSync(`matches-${name}-meta.json`, JSON.stringify(meta, null, 2), "utf-8");
  
  return meta;
}

// Yaklaşan etkinlikleri çeken fonksiyon
async function scrapeUpcoming(siteConfig) {
  const { name, domain, subDomain } = siteConfig;
  const url = `${subDomain}.${domain}`;
  
  const browser = await chromium.launch({ headless: true });
  const context = await browser.newContext({
    ...iPhone,
    userAgent:
      "Mozilla/5.0 (iPhone; CPU iPhone OS 15_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/15.0 Mobile/15E148 Safari/604.1",
  });
  const page = await context.newPage();
  
  await page.goto(`https://${url}/tr/`);
  
  // Önce yaklaşan etkinlikler tab'ına tıkla
  try {
    await page.click('div.OM-Slider__Action:has-text("yaklaşan Karşılaşmalar")', { timeout: 5000 });
    await page.waitForTimeout(1000); // Tab geçişi için bekle
  } catch (e) {
    console.log(`⚠️  Yaklaşan etkinlikler tab'ı bulunamadı, direkt selector ile devam ediliyor...`);
  }
  
  // Yaklaşan etkinlikler bölümünü bekle
  await page.waitForSelector('div.MatchList.MatchList--Sport-1.MatchList--Outcomes-1.MatchList--UpcomingEvents', { timeout: 15000 });
  
  let content = await page.locator('div.MatchList.MatchList--Sport-1.MatchList--Outcomes-1.MatchList--UpcomingEvents').innerHTML();
  
  // HTML içinde hem www hem de subdomain linklerini replace et
  const regexWww = new RegExp(`https://www\\.${domain.replace(/\./g, '\\.')}/`, "g");
  const regexSubdomain = new RegExp(`https://${subDomain.replace(/\./g, '\\.')}\\.${domain.replace(/\./g, '\\.')}/`, "g");
  
  content = content.replace(regexWww, "/");
  content = content.replace(regexSubdomain, "/");
  
  await browser.close();
  
  const fileName = `upcoming-matches-${name}.html`;
  fs.writeFileSync(fileName, content, "utf-8");
  
  const meta = { 
    lastFetched: new Date().toISOString(),
    site: name,
    domain: url,
    type: "upcoming"
  };
  fs.writeFileSync(`upcoming-matches-${name}-meta.json`, JSON.stringify(meta, null, 2), "utf-8");
  
  return meta;
}

// Dinamik olarak her site için endpoint oluştur
sites.forEach(site => {
  const siteName = site.name;
  
  // /{siteName}/matches endpoint
  app.get(`/${siteName}/matches`, async (req, res) => {
    try {
      const data = await scrape(site);
      res.json(data);
    } catch (e) {
      res.status(500).json({ error: e.message });
    }
  });
  
  // /{siteName}/matches.html endpoint
  app.get(`/${siteName}/matches.html`, (req, res) => {
    const filePath = path.join(process.cwd(), `matches-${siteName}.html`);
    if (fs.existsSync(filePath)) {
      res.sendFile(filePath);
    } else {
      res.status(404).send(`HTML dosyası bulunamadı. Önce /${siteName}/matches endpoint'ini çalıştırın.`);
    }
  });
  
  // /{siteName}/lastfetched endpoint
  app.get(`/${siteName}/lastfetched`, (req, res) => {
    const metaPath = path.join(process.cwd(), `matches-${siteName}-meta.json`);
    if (fs.existsSync(metaPath)) {
      const meta = JSON.parse(fs.readFileSync(metaPath, "utf-8"));
      res.json(meta);
    } else {
      res.status(404).json({ error: "Henüz veri çekilmedi." });
    }
  });
  
  // /{siteName}/upcoming-matches endpoint
  app.get(`/${siteName}/upcoming-matches`, async (req, res) => {
    try {
      const data = await scrapeUpcoming(site);
      res.json(data);
    } catch (e) {
      res.status(500).json({ error: e.message });
    }
  });
  
  // /{siteName}/upcoming-matches.html endpoint
  app.get(`/${siteName}/upcoming-matches.html`, (req, res) => {
    const filePath = path.join(process.cwd(), `upcoming-matches-${siteName}.html`);
    if (fs.existsSync(filePath)) {
      res.sendFile(filePath);
    } else {
      res.status(404).send(`HTML dosyası bulunamadı. Önce /${siteName}/upcoming-matches endpoint'ini çalıştırın.`);
    }
  });
  
  // /{siteName}/upcoming-lastfetched endpoint
  app.get(`/${siteName}/upcoming-lastfetched`, (req, res) => {
    const metaPath = path.join(process.cwd(), `upcoming-matches-${siteName}-meta.json`);
    if (fs.existsSync(metaPath)) {
      const meta = JSON.parse(fs.readFileSync(metaPath, "utf-8"));
      res.json(meta);
    } else {
      res.status(404).json({ error: "Henüz veri çekilmedi." });
    }
  });
});

app.listen(PORT, () => {
  console.log(`✅ API çalışıyor: http://localhost:${PORT}`);
  console.log(`📋 Yüklenen siteler: ${sites.map(s => s.name).join(", ")}`);
});