import express from "express";
import { chromium, devices } from "playwright";
import fs from "fs";
import path from "path";
import axios from "axios";
import https from "https";

const httpsAgent = new https.Agent({ rejectUnauthorized: false });

const app = express();
const PORT = 3131;

// CORS middleware
app.use((req, res, next) => {
  res.header("Access-Control-Allow-Origin", "*");
  res.header("Access-Control-Allow-Methods", "GET, POST, PUT, DELETE, OPTIONS");
  res.header("Access-Control-Allow-Headers", "Origin, X-Requested-With, Content-Type, Accept, Authorization, Cache-Control, Pragma");
  res.header("Access-Control-Allow-Credentials", "true");
  
  if (req.method === "OPTIONS") {
    res.sendStatus(200);
  } else {
    next();
  }
});

// Body parsing middleware'leri
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.text());
app.use(express.raw());

const iPhone = devices["iPhone 13"];

async function scrape() {
  const browser = await chromium.launch({ headless: true });
  const context = await browser.newContext({
    ...iPhone,
    userAgent:
      "Mozilla/5.0 (iPhone; CPU iPhone OS 15_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/15.0 Mobile/15E148 Safari/604.1",
  });

  const page = await context.newPage();
  await page.goto("https://sports2.holiganbet1148.com/tr/");

  await page.waitForSelector(
    "div.MatchList.MatchList--Sport-1.MatchList--Outcomes-1.MatchList--LiveEvents",
    { timeout: 15000 }
  );

  let content = await page
    .locator(
      "div.MatchList.MatchList--Sport-1.MatchList--Outcomes-1.MatchList--LiveEvents"
    )
    .innerHTML();

  // HTML içinde tüm 'https://www.holiganbet1148.com/' ifadelerini '/' ile değiştir
  content = content.replace(/https:\/\/www\.holiganbet1148\.com\//g, "/");

  await browser.close();

  fs.writeFileSync("matches.html", content, "utf-8");

  const meta = { lastFetched: new Date().toISOString() };
  fs.writeFileSync("matches-meta.json", JSON.stringify(meta, null, 2), "utf-8");

  return meta;
}

// /matches endpoint’i: HTML’i scrape edip meta JSON döner
app.get("/matches", async (req, res) => {
  try {
    const data = await scrape();
    res.json(data);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// /matches.html endpoint’i: HTML dosyasını direkt gönderir
app.get("/matches.html", (req, res) => {
  const filePath = path.join(process.cwd(), "matches.html");
  if (fs.existsSync(filePath)) {
    res.sendFile(filePath);
  } else {
    res.status(404).send("HTML dosyası bulunamadı. Önce /matches endpoint'ini çalıştırın.");
  }
});

// /lastfetched endpoint’i: son çekilme zamanını gösterir
app.get("/lastfetched", (req, res) => {
  const metaPath = path.join(process.cwd(), "matches-meta.json");
  if (fs.existsSync(metaPath)) {
    const meta = JSON.parse(fs.readFileSync(metaPath, "utf-8"));
    res.json(meta);
  } else {
    res.status(404).json({ error: "Henüz veri çekilmedi." });
  }
});

// Domain replace fonksiyonu
function replaceBlockedDomains(content, contentType) {
  if (contentType && contentType.includes("text")) {
    // demogamesfree.avwyktejpg.net domain'ini kendi sunucumuza yönlendir
    content = content.replace(/demogamesfree\.avwyktejpg\.net/g, "lain.com.tr/demogame");
    // Diğer engellenmiş domain'ler varsa buraya eklenebilir
  }
  return content;
}

// Proxy: sadece /game/* path'lerini gamelaunch.holiganbet1148.com'a yönlendir
app.use("/game", async (req, res) => {
  try {
    // /game sonrası path
    const targetPath = req.originalUrl.replace(/^\/game/, ""); 
    const targetUrl = `https://gamelaunch.holiganbet1148.com${targetPath}`;

    console.log(`🎮 Game Request: ${req.method} ${req.originalUrl} -> ${targetUrl}`);

    res.setHeader("Access-Control-Allow-Origin", "*");
    res.setHeader("Access-Control-Allow-Methods", "GET,POST,PUT,DELETE,OPTIONS");
    res.setHeader("Access-Control-Allow-Headers", "*");

    // Request konfigürasyonu - tüm HTTP metodlarını destekle
    const requestConfig = {
      method: req.method.toLowerCase(),
      url: targetUrl,
      headers: {
        // Önce orijinal header'ları kopyala
        ...req.headers,
        // Sonra önemli header'ları override et
        Referer: "https://gamelaunch.holiganbet1148.com",
        "User-Agent":
          "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/140.0.0.0 Safari/537.36",
        Host: "gamelaunch.holiganbet1148.com",
      },
      httpsAgent,
    };

    // POST, PUT, PATCH için body'yi ekle
    if (['post', 'put', 'patch'].includes(req.method.toLowerCase())) {
      requestConfig.data = req.body;
    }

    // Query parametrelerini ekle
    if (Object.keys(req.query).length > 0) {
      requestConfig.params = req.query;
    }

    const response = await axios(requestConfig);

    const contentType = response.headers["content-type"] || "text/plain";
    let responseData = response.data;

    // Response içeriğini kontrol et ve engellenmiş domain'leri replace et
    if (typeof responseData === "string") {
      responseData = replaceBlockedDomains(responseData, contentType);
    }

    // Response header'larını kopyala
    Object.keys(response.headers).forEach(key => {
      if (key.toLowerCase() !== 'content-encoding' && key.toLowerCase() !== 'transfer-encoding') {
        res.set(key, response.headers[key]);
      }
    });

    // Content-Type'ı override et
    res.set("Content-Type", contentType);
    
    // Status code'u koru
    res.status(response.status);
    res.send(responseData);
  } catch (err) {
    console.error("❌ Game Proxy Error:", err.message);
    res.status(500).send("Proxy error");
  }
});

// Demo game proxy: demogamesfree.avwyktejpg.net için ayrı endpoint
app.use("/demogame", async (req, res) => {
  try {
    // /demogame sonrası path
    const targetPath = req.originalUrl.replace(/^\/demogame/, ""); 
    const targetUrl = `https://demogamesfree.avwyktejpg.net${targetPath}`;

    console.log(`🎯 Demo Game Request: ${req.method} ${req.originalUrl} -> ${targetUrl}`);

    res.setHeader("Access-Control-Allow-Origin", "*");
    res.setHeader("Access-Control-Allow-Methods", "GET,POST,PUT,DELETE,OPTIONS");
    res.setHeader("Access-Control-Allow-Headers", "*");

    // Request konfigürasyonu - tüm HTTP metodlarını destekle
    const requestConfig = {
      method: req.method.toLowerCase(),
      url: targetUrl,
      headers: {
        // Önce orijinal header'ları kopyala
        ...req.headers,
        // Sonra önemli header'ları override et
        Referer: "https://demogamesfree.avwyktejpg.net",
        "User-Agent":
          "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/140.0.0.0 Safari/537.36",
        Host: "demogamesfree.avwyktejpg.net",
      },
      httpsAgent,
    };

    // POST, PUT, PATCH için body'yi ekle
    if (['post', 'put', 'patch'].includes(req.method.toLowerCase())) {
      requestConfig.data = req.body;
    }

    // Query parametrelerini ekle
    if (Object.keys(req.query).length > 0) {
      requestConfig.params = req.query;
    }

    const response = await axios(requestConfig);

    const contentType = response.headers["content-type"] || "text/plain";
    let responseData = response.data;

    // Response içeriğini kontrol et ve domain'leri kendi sunucumuza yönlendir
    if (typeof responseData === "string") {
      // demogamesfree.avwyktejpg.net'i kendi sunucumuza yönlendir
      responseData = responseData.replace(/demogamesfree\.avwyktejpg\.net/g, "lain.com.tr/demogame");
      // Diğer engellenmiş domain'ler varsa buraya eklenebilir
    }

    // Response header'larını kopyala
    Object.keys(response.headers).forEach(key => {
      if (key.toLowerCase() !== 'content-encoding' && key.toLowerCase() !== 'transfer-encoding') {
        res.set(key, response.headers[key]);
      }
    });

    // Content-Type'ı override et
    res.set("Content-Type", contentType);
    
    // Status code'u koru
    res.status(response.status);
    res.send(responseData);
  } catch (err) {
    console.error("❌ Demo Game Proxy Error:", err.message);
    res.status(500).send("Demo game proxy error");
  }
});

// GS2C proxy: demogamesfree.avwyktejpg.net/gs2c için ayrı endpoint
app.use("/gs2c", async (req, res) => {
  try {
    // /gs2c sonrası path
    const targetPath = req.originalUrl.replace(/^\/gs2c/, ""); 
    const targetUrl = `https://demogamesfree.avwyktejpg.net/gs2c${targetPath}`;

    console.log(`🔧 GS2C Request: ${req.method} ${req.originalUrl} -> ${targetUrl}`);

    res.setHeader("Access-Control-Allow-Origin", "*");
    res.setHeader("Access-Control-Allow-Methods", "GET,POST,PUT,DELETE,OPTIONS");
    res.setHeader("Access-Control-Allow-Headers", "*");

    // Request konfigürasyonu - tüm HTTP metodlarını destekle
    const requestConfig = {
      method: req.method.toLowerCase(),
      url: targetUrl,
      headers: {
        // Önce orijinal header'ları kopyala
        ...req.headers,
        // Sonra önemli header'ları override et
        Referer: "https://demogamesfree.avwyktejpg.net",
        "User-Agent":
          "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/140.0.0.0 Safari/537.36",
        Host: "demogamesfree.avwyktejpg.net",
      },
      httpsAgent,
    };

    // POST, PUT, PATCH için body'yi ekle
    if (['post', 'put', 'patch'].includes(req.method.toLowerCase())) {
      requestConfig.data = req.body;
    }

    // Query parametrelerini ekle
    if (Object.keys(req.query).length > 0) {
      requestConfig.params = req.query;
    }

    const response = await axios(requestConfig);

    const contentType = response.headers["content-type"] || "text/plain";
    let responseData = response.data;

    // Response içeriğini kontrol et ve domain'leri kendi sunucumuza yönlendir
    if (typeof responseData === "string") {
      // demogamesfree.avwyktejpg.net'i kendi sunucumuza yönlendir
      responseData = responseData.replace(/demogamesfree\.avwyktejpg\.net/g, "lain.com.tr/gs2c");
      // Diğer engellenmiş domain'ler varsa buraya eklenebilir
    }

    // Response header'larını kopyala
    Object.keys(response.headers).forEach(key => {
      if (key.toLowerCase() !== 'content-encoding' && key.toLowerCase() !== 'transfer-encoding') {
        res.set(key, response.headers[key]);
      }
    });

    // Content-Type'ı override et
    res.set("Content-Type", contentType);
    
    // Status code'u koru
    res.status(response.status);
    res.send(responseData);
  } catch (err) {
    console.error("❌ GS2C Proxy Error:", err.message);
    res.status(500).send("GS2C proxy error");
  }
});


app.listen(PORT, () =>
  console.log(`✅ API çalışıyor: http://localhost:${PORT}`)
);
