/**
 * Game Launcher JavaScript
 * Oyun başlatma ve kontrol sistemi
 */

// Oyun listesi artık kullanılmıyor - direkt game-id ile oyun açılıyor

// Seçili oyun ID'sini sakla
let selectedGameId = '';
let isLiveCasino = false;

/**
 * Kullanıcının giriş yapıp yapmadığını kontrol eder
 * @return {boolean} True eğer giriş yapmışsa
 */
function isUserLoggedIn() {
    // PHP'den gelen giriş durumunu kontrol et
    return window.userLoggedIn === true;
}

/**
 * Giriş gerekli modal'ını göster
 */
function showLoginRequiredModal() {
    const modal = document.getElementById('loginRequiredModal');
    if (modal) {
        modal.style.display = 'block';
        document.body.style.overflow = 'hidden'; // Scroll'u engelle
    }
}

/**
 * Giriş gerekli modal'ını kapat
 */
function closeLoginRequiredModal() {
    const modal = document.getElementById('loginRequiredModal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto'; // Scroll'u geri aç
    }
}

/**
 * Giriş modal'ını aç
 */
function openLoginModal() {
    closeLoginRequiredModal();
    const loginModal = document.getElementById('loginModal');
    if (loginModal) {
        loginModal.style.display = 'block';
        document.body.style.overflow = 'hidden';
    }
}

/**
 * Casino oyunu başlat (Demo/Para Yatır modal'ı)
 * @param {string} gameId - Oyun ID'si
 */
function launchGame(gameId) {
    selectedGameId = gameId;
    isLiveCasino = false;
    
    // Önce giriş kontrolü yap
    if (!isUserLoggedIn()) {
        showLoginRequiredModal();
        return;
    }
    
    showBalanceModal();
}

/**
 * Live casino oyunu başlat (Sadece Para Yatır modal'ı)
 * @param {string} gameId - Oyun ID'si
 */
function launchLiveCasino(gameId) {
    selectedGameId = gameId;
    isLiveCasino = true;
    
    // Önce giriş kontrolü yap
    if (!isUserLoggedIn()) {
        showLoginRequiredModal();
        return;
    }
    
    showLiveCasinoModal();
}

/**
 * Demo oyun başlat
 */
function playDemo() {
    // Seçili oyunu direkt aç
    // Eğer direct game ise, game-id parametresi ile aç
    // window.location.href = '/?page=game&game-id=' + selectedGameId;
    window.location.href = '/game/' + selectedGameId;
}

/**
 * Bakiye yetersiz modal'ını göster
 */
function showBalanceModal() {
    const modal = document.getElementById('balanceModal');
    if (modal) {
        // Bakiye bilgisini güncelle
        updateBalanceInModal();
        
        modal.style.display = 'block';
        document.body.style.overflow = 'hidden'; // Scroll'u engelle
    }
}

/**
 * Modal'daki bakiye bilgisini güncelle
 */
function updateBalanceInModal() {
    const balanceElement = document.querySelector('#balanceModal .FormattedAmount');
    
    if (!balanceElement) return;
    
    // Önce global değişkenden bakiye al
    if (window.currentUserBalance) {
        balanceElement.textContent = window.currentUserBalance;
        return;
    }
    
    // Global değişken yoksa AJAX ile güncel bakiye bilgisini al
    fetch('/api/get-balance.php')
        .then(response => response.json())
        .then(data => {
            if (data.balance !== undefined) {
                balanceElement.textContent = data.balance;
            }
        })
        .catch(error => {
            console.log('Bakiye bilgisi alınamadı:', error);
            // Hata durumunda default değer
            balanceElement.textContent = '0,00';
        });
}

/**
 * Live casino modal'ını göster
 */
function showLiveCasinoModal() {
    const modal = document.getElementById('liveCasinoModal');
    if (modal) {
        // Bakiye bilgisini güncelle
        updateLiveCasinoBalance();
        
        modal.style.display = 'block';
        document.body.style.overflow = 'hidden'; // Scroll'u engelle
    }
}

/**
 * Live casino modal'ındaki bakiye bilgisini güncelle
 */
function updateLiveCasinoBalance() {
    const balanceElement = document.getElementById('liveCasinoBalance');
    
    if (!balanceElement) return;
    
    // Önce global değişkenden bakiye al
    if (window.currentUserBalance) {
        balanceElement.textContent = window.currentUserBalance;
        return;
    }
    
    // Global değişken yoksa AJAX ile güncel bakiye bilgisini al
    fetch('/api/get-balance.php')
        .then(response => response.json())
        .then(data => {
            if (data.balance !== undefined) {
                balanceElement.textContent = data.balance;
            }
        })
        .catch(error => {
            console.log('Live casino bakiye bilgisi alınamadı:', error);
            // Hata durumunda default değer
            balanceElement.textContent = '0,00';
        });
}

/**
 * Live casino modal'ını kapat
 */
function closeLiveCasinoModal() {
    const modal = document.getElementById('liveCasinoModal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto'; // Scroll'u geri aç
    }
}

/**
 * Modal'ı kapat
 */
function closeModal() {
    const modal = document.getElementById('balanceModal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto'; // Scroll'u geri aç
    }
}

/**
 * Modal overlay'e tıklandığında kapat
 */
function closeModalOnOverlay(event) {
    if (event.target.classList.contains('ModalOverlay')) {
        closeModal();
    }
}

/**
 * Live casino modal overlay'e tıklandığında kapat
 */
function closeLiveCasinoModalOnOverlay(event) {
    if (event.target.classList.contains('ModalOverlay')) {
        closeLiveCasinoModal();
    }
}

// DOM yüklendiğinde event listener'ları ekle
document.addEventListener('DOMContentLoaded', function() {
    // Casino modal overlay click event
    const modal = document.getElementById('balanceModal');
    if (modal) {
        modal.addEventListener('click', closeModalOnOverlay);
    }
    
    // Live casino modal overlay click event
    const liveCasinoModal = document.getElementById('liveCasinoModal');
    if (liveCasinoModal) {
        liveCasinoModal.addEventListener('click', closeLiveCasinoModalOnOverlay);
    }
    
    // ESC tuşu ile modal kapatma
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            // Hangi modal açıksa onu kapat
            if (document.getElementById('balanceModal').style.display === 'block') {
                closeModal();
            }
            if (document.getElementById('liveCasinoModal').style.display === 'block') {
                closeLiveCasinoModal();
            }
        }
    });
});
