<?php
/**
 * Payment System Functions
 * Functions for IBAN, payment methods, and approval system
 */

// Include database configuration
require_once __DIR__ . '/database.php';

include 'config.php';

// Initialize database on first load
try {
    createTablesIfNotExist();
} catch (Exception $e) {
    error_log("Database initialization failed: " . $e->getMessage());
}

/**
 * Get IBAN information by bank ID
 * @param string $bankId
 * @return array|null
 */
function getIban($bankId) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT * FROM bank_accounts WHERE bank_id = ? AND is_active = 1");
        $stmt->execute([$bankId]);
        $result = $stmt->fetch();
        
        if ($result) {
            return [
                'bank_id' => $result['bank_id'],
                'bank_name' => $result['bank_name'],
                'bank_logo' => $result['bank_logo'],
                'account_holder' => $result['account_holder'],
                'branch_code' => $result['branch_code'],
                'account_number' => $result['account_number'],
                'iban' => $result['iban'],
                'min_amount' => number_format($result['min_amount'], 2),
                'max_amount' => number_format($result['max_amount'], 2),
                'currency' => $result['currency']
            ];
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Failed to get IBAN data: " . $e->getMessage());
        return null;
    }
}

/**
 * Get all available banks
 * @return array
 */
function getAllBanks() {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT bank_id as id, bank_name as name FROM bank_accounts WHERE is_active = 1 ORDER BY bank_name");
        $stmt->execute();
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Failed to get banks: " . $e->getMessage());
        return [];
    }
}

/**
 * Get payment method information
 * @param string $paymentId
 * @return array|null
 */
function getPaymentMethod($paymentId) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT * FROM payment_methods WHERE method_id = ? AND is_active = 1");
        $stmt->execute([$paymentId]);
        $result = $stmt->fetch();
        
        if ($result) {
            return [
                'id' => $result['method_id'],
                'name' => $result['name'],
                'logo' => $result['logo'],
                'category' => $result['category'],
                'iban_supported' => (bool)$result['iban_supported']
            ];
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Failed to get payment method: " . $e->getMessage());
        return null;
    }
}

/**
 * Format amount for display
 * @param float $amount
 * @param string $currency
 * @return string
 */
function formatAmount($amount, $currency = 'TRY') {
    return number_format($amount, 2, ',', '.') . ' ' . $currency;
}

/**
 * Generate CSRF token
 * @return string
 */
function generateCSRFToken() {
    return bin2hex(random_bytes(32));
}

/**
 * Validate bank ID
 * @param string $bankId
 * @return bool
 */
function isValidBankId($bankId) {
    $validBanks = ['1', '2', '3', '4', '5', '11'];
    return in_array($bankId, $validBanks);
}

/**
 * Get user information (mock data - in real app this would come from session/database)
 * @return array
 */
function getUserInfo() {
    return [
        'name' => '',
        'username' => '',
        'email' => 'ahmet@example.com'
    ];
}

/**
 * Process payment approval
 * @param array $paymentData
 * @return array
 */
function processPaymentApproval($paymentData) {
    // Validate required fields
    $requiredFields = ['bank_id', 'payment_id', 'amount', 'user_name', 'user_surname', 'username'];
    foreach ($requiredFields as $field) {
        if (empty($paymentData[$field])) {
            return [
                'success' => false,
                'message' => "Required field missing: {$field}",
                'transaction_id' => null
            ];
        }
    }

    // Generate transaction ID
    $transactionId = 'TXN_' . time() . '_' . rand(1000, 9999);
    
    // Get IBAN and payment info
    $ibanInfo = getIban($paymentData['bank_id']);
    $paymentInfo = getPaymentMethod($paymentData['payment_id']);
    
    if (!$ibanInfo) {
        return [
            'success' => false,
            'message' => 'Invalid bank ID',
            'transaction_id' => null
        ];
    }

    // Prepare transaction data
    $transactionData = [
        'transaction_id' => $transactionId,
        'user_name' => $paymentData['user_name'],
        'user_surname' => $paymentData['user_surname'] ?? '',
        'username' => $paymentData['username'],
        'amount' => $paymentData['amount'],
        'bank_id' => $paymentData['bank_id'],
        'bank_name' => $ibanInfo['bank_name'],
        'iban' => $ibanInfo['iban'],
        'account_holder' => $ibanInfo['account_holder'],
        'payment_method' => $paymentInfo ? $paymentInfo['name'] : 'Unknown',
        'status' => 'pending',
        'created_at' => date('Y-m-d H:i:s'),
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ];

    // In a real application, you would:
    // 1. Save to database
    // 2. Send notification to admin
    // 3. Log the transaction
    // 4. Send confirmation email/SMS
    // 5. Integrate with payment gateway
    
    // For now, we'll just log it and simulate success
    error_log("Payment approval processed: " . json_encode($transactionData));
    
    // Simulate database save
    $saved = saveTransactionToDatabase($transactionData);
    
    if ($saved) {
        // Send admin notification
        sendAdminNotification($transactionData);
        
        // Send user confirmation
        sendUserConfirmation($transactionData);
        
        return [
            'success' => true,
            'message' => 'Payment approval submitted successfully',
            'transaction_id' => $transactionId,
            'data' => $transactionData
        ];
    } else {
        return [
            'success' => false,
            'message' => 'Failed to save transaction',
            'transaction_id' => null
        ];
    }
}

/**
 * Save transaction to database
 * @param array $transactionData
 * @return bool
 */
function saveTransactionToDatabase($transactionData) {
    try {
        $pdo = getDatabaseConnection();
        $sql = "INSERT INTO transactions (
            transaction_id, user_name, user_surname, username, amount, 
            payment_method_id, bank_id, crypto_type, crypto_address, 
            crypto_amount, status, ip_address, user_agent
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            $transactionData['transaction_id'],
            $transactionData['user_name'],
            $transactionData['user_surname'] ?? '',
            $transactionData['username'],
            $transactionData['amount'],
            $transactionData['payment_method'] ?? '',
            $transactionData['bank_id'] ?? null,
            $transactionData['crypto_type'] ?? null,
            $transactionData['crypto_address'] ?? null,
            $transactionData['crypto_amount'] ?? null,
            $transactionData['status'],
            $transactionData['ip_address'],
            $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
        
        return $result;
    } catch (Exception $e) {
        error_log("Failed to save transaction: " . $e->getMessage());
        return false;
    }
}

/**
 * Send admin notification (mock function)
 * @param array $transactionData
 * @return bool
 */
function sendAdminNotification($transactionData) {
    // In real application, this would send email/SMS to admin
    error_log("Admin notification sent for transaction: " . $transactionData['transaction_id']);
    return true;
}

/**
 * Send user confirmation (mock function)
 * @param array $transactionData
 * @return bool
 */
function sendUserConfirmation($transactionData) {
    // In real application, this would send email/SMS to user
    error_log("User confirmation sent for transaction: " . $transactionData['transaction_id']);
    return true;
}

/**
 * Get transaction by ID
 * @param string $transactionId
 * @return array|null
 */
function getTransactionById($transactionId) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT * FROM transactions WHERE transaction_id = ?");
        $stmt->execute([$transactionId]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log("Failed to get transaction: " . $e->getMessage());
        return null;
    }
}

/**
 * Get crypto address for specific cryptocurrency
 * @param string $cryptoType (btc, usdt, trx)
 * @return array|null
 */
function getCryptoAddress($cryptoType) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT * FROM crypto_addresses WHERE crypto_type = ? AND is_active = 1");
        $stmt->execute([$cryptoType]);
        $result = $stmt->fetch();
        
        if ($result) {
    return [
                'type' => $result['crypto_type'],
                'name' => $result['name'],
                'address' => $result['address'],
                'network' => $result['network'],
                'min_amount' => $result['min_amount'],
                'max_amount' => $result['max_amount'],
                'icon' => $result['icon'],
                'exchange_rate' => $result['exchange_rate']
            ];
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Failed to get crypto address: " . $e->getMessage());
        return null;
    }
}

/**
 * Get all available crypto types
 * @return array
 */
function getAllCryptoTypes() {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT crypto_type as id, name, icon FROM crypto_addresses WHERE is_active = 1 ORDER BY crypto_type");
        $stmt->execute();
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Failed to get crypto types: " . $e->getMessage());
        return [];
    }
}

/**
 * Calculate crypto amount from TL amount
 * @param float $tlAmount
 * @param string $cryptoType
 * @return float
 */
function calculateCryptoAmount($tlAmount, $cryptoType) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT exchange_rate FROM crypto_addresses WHERE crypto_type = ? AND is_active = 1");
        $stmt->execute([$cryptoType]);
        $result = $stmt->fetch();
        
        if ($result && $result['exchange_rate'] > 0) {
            return $tlAmount / $result['exchange_rate'];
        }
        
        return 0;
    } catch (Exception $e) {
        error_log("Failed to calculate crypto amount: " . $e->getMessage());
        return 0;
    }
}

/**
 * Format crypto amount for display
 * @param float $amount
 * @param string $cryptoType
 * @return string
 */
function formatCryptoAmount($amount, $cryptoType) {
    $decimals = [
        'btc' => 7,
        'usdt' => 2,
        'trx' => 0
    ];

    $decimalPlaces = isset($decimals[$cryptoType]) ? $decimals[$cryptoType] : 2;
    return number_format($amount, $decimalPlaces, ',', '.');
}

/**
 * Get valid payment methods grouped by category
 * @return array
 */
function getValidPaymentMethods() {
    $paymentMethods = [
        'bank_transfer' => [
            'name' => 'Banka Havalesi (IBAN)',
            'description' => 'Banka havalesi ile ödeme yapın',
            'methods' => [
                'ekofix-deposit-bank-transfer' => 'Ekofix Deposit - Bank Transfer',
                'ngspay-deposit' => 'NGS Pay Deposit',
                'anindabankav2-deposit' => 'Anında Banka Deposit',
                'garanti-bankasi-deposit-transfer' => 'Garanti Bankasi Para Yatırım',
                'akbank-deposit-transfer' => 'Akbank Para Yatırım',
                'isbank-deposit-transfer' => 'Is Bankasi Para Yatırım',
                'ziraat-deposit-transfer' => 'Ziraat Bankasi Para Yatırım',
                'yapikredi-deposit-transfer' => 'Yapi Kredi Para Yatırım'
            ]
        ],
        'crypto' => [
            'name' => 'Kripto Para',
            'description' => 'Kripto para ile ödeme yapın',
            'methods' => [
                'anindabtcv2-deposit' => 'Anında Kripto Deposit',
                'paytu-crypto-deposit' => 'Payyu Crypto Deposit',
                'upaycoins-deposit' => 'Youpaycoin Yatırım'
            ]
        ]
    ];

    return $paymentMethods;
}

/**
 * Telegram mesaj gönderme fonksiyonu
 * @param string $message
 * @param string $chatId (optional)
 * @return bool
 */
function sendTelegramMessage($message, $chatId = null) {
    // Önce admin sistemindeki Telegram ayarlarını kontrol et
    if (function_exists('getTelegramSettings')) {
        try {
            $telegramSettings = getTelegramSettings();
            
            if (!$telegramSettings['enabled'] || $telegramSettings['bot_token'] === 'YOUR_BOT_TOKEN_HERE') {
                return false;
            }
            
            $botToken = $telegramSettings['bot_token'];
            $targetChatId = $chatId ?: $telegramSettings['login_chat_id'];
            
            if ($targetChatId === 'YOUR_CHAT_ID_HERE') {
                return false;
            }
            
            $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
            
            $data = [
                'chat_id' => $targetChatId,
                'text' => $message,
                'parse_mode' => 'HTML'
            ];
            
            $options = [
                'http' => [
                    'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                    'method' => 'POST',
                    'content' => http_build_query($data)
                ]
            ];
            
            $context = stream_context_create($options);
            $result = file_get_contents($url, false, $context);
            
            if ($result === FALSE) {
                error_log("Telegram message failed to send");
                return false;
            }
            
            $response = json_decode($result, true);
            return $response['ok'] ?? false;
            
        } catch (Exception $e) {
            error_log("Failed to send Telegram message via admin system: " . $e->getMessage());
        }
    }
    
    // Eski sistem (telegram.php) - fallback
    if (file_exists(__DIR__ . '/telegram.php')) {
        require_once __DIR__ . '/telegram.php';
        
        // Telegram aktif değilse çık
        if (!TELEGRAM_ENABLED) {
            error_log("Telegram mesaj gönderimi devre dışı!");
            return false;
        }
        
        $botToken = TELEGRAM_BOT_TOKEN;
        $targetChatId = $chatId ?: TELEGRAM_CHAT_ID;
        
        // Bot token ve chat ID kontrolü
        if ($botToken === 'YOUR_BOT_TOKEN_HERE' || $targetChatId === 'YOUR_CHAT_ID_HERE') {
            error_log("Telegram bot token veya chat ID ayarlanmamış!");
            return false;
        }
        
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        
        $data = [
            'chat_id' => $targetChatId,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        $result = file_get_contents($url, false, $context);
        
        if ($result === FALSE) {
            error_log("Telegram mesajı gönderilemedi!");
            return false;
        }
        
        $response = json_decode($result, true);
        return $response['ok'] ?? false;
    }
    
    return false;
}

/**
 * IBAN ödeme onayı için Telegram mesajı oluştur
 */
function createIbanPaymentTelegramMessage($paymentData) {
    $bankName = $paymentData['bank_name'] ?? 'Bilinmeyen Banka';
    $amount = number_format($paymentData['amount'], 2, ',', '.') . ' TL';
    $userName = $paymentData['user_name'] ?? '';
    $userSurname = $paymentData['user_surname'] ?? '';
    $username = $paymentData['username'] ?? '';
    $iban = $paymentData['iban'] ?? '';
    $accountHolder = $paymentData['account_holder'] ?? '';
    $transactionId = $paymentData['transaction_id'] ?? '';
    $date = date('d.m.Y H:i:s');
    
    $message = "🏦 <b>IBAN ÖDEME ONAYI</b>\n\n";
    $message .= "💰 <b>Tutar:</b> {$amount}\n";
    $message .= "🏛️ <b>Banka:</b> {$bankName}\n";
    $message .= "👤 <b>Kullanıcı:</b> {$userName} {$userSurname}\n";
    $message .= "🔑 <b>Kullanıcı Adı:</b> {$username}\n";
    $message .= "📋 <b>IBAN:</b> <code>{$iban}</code>\n";
    $message .= "👥 <b>Hesap Sahibi:</b> {$accountHolder}\n";
    $message .= "🆔 <b>İşlem ID:</b> {$transactionId}\n";
    $message .= "📅 <b>Tarih:</b> {$date}\n\n";
    $message .= "✅ <b>Ödeme onaylandı ve işleme alındı!</b>";
    
    return $message;
}

/**
 * Kripto ödeme onayı için Telegram mesajı oluştur
 */
function createCryptoPaymentTelegramMessage($paymentData) {
    $cryptoType = strtoupper($paymentData['crypto_type'] ?? 'Bilinmeyen');
    $amount = $paymentData['amount'] ?? '';
    $cryptoAmount = $paymentData['crypto_amount'] ?? '';
    $userName = $paymentData['user_name'] ?? '';
    $userSurname = $paymentData['user_surname'] ?? '';
    $username = $paymentData['username'] ?? '';
    $address = $paymentData['crypto_address'] ?? '';
    $transactionId = $paymentData['transaction_id'] ?? '';
    $date = date('d.m.Y H:i:s');
    
    $message = "₿ <b>KRİPTO ÖDEME ONAYI</b>\n\n";
    $message .= "💰 <b>Tutar:</b> {$amount} TL\n";
    $message .= "₿ <b>Kripto:</b> {$cryptoType}\n";
    $message .= "📊 <b>Kripto Miktarı:</b> {$cryptoAmount} {$cryptoType}\n";
    $message .= "👤 <b>Kullanıcı:</b> {$userName} {$userSurname}\n";
    $message .= "🔑 <b>Kullanıcı Adı:</b> {$username}\n";
    $message .= "🔗 <b>Adres:</b> <code>{$address}</code>\n";
    $message .= "🆔 <b>İşlem ID:</b> {$transactionId}\n";
    $message .= "📅 <b>Tarih:</b> {$date}\n\n";
    $message .= "✅ <b>Ödeme onaylandı ve işleme alındı!</b>";
    
    return $message;
}


session_start();

/**
 * Kullanıcı giriş yaptı mı kontrol eder
 * @return bool
 */
function isLoggedIn() {
    return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

/**
 * Giriş yapan kullanıcının kullanıcı adını döner
 * @return string|null
 */
function getLoggedInUsername() {
    return $_SESSION['username'] ?? null;
}

function showMatchesFromApi() {
    global $bet_node;
    $lastFetchedUrl = $bet_node . "/lastfetched";
    $matchesEndpoint = $bet_node . "/matches";
    $matchesHtmlEndpoint = $bet_node . "/matches.html";

    // lastfetched zamanını al
    $lastFetchedData = file_get_contents($lastFetchedUrl);
    $lastFetched = null;

    if ($lastFetchedData !== false) {
        $json = json_decode($lastFetchedData, true);
        if (isset($json['lastFetched'])) {
            $lastFetched = strtotime($json['lastFetched']);
        }
    }

    $oneHourAgo = time() - 600;

    // 1 saatten eskiyse /matches endpoint'ini çağır ve veri güncelle
    if ($lastFetched === null || $lastFetched < $oneHourAgo) {
        $update = file_get_contents($matchesEndpoint);
        if ($update === false) {
            die("Veri çekilemedi.");
        }
    }

    // API'den güncel HTML'i çek ve yansıt
    $htmlContent = file_get_contents($matchesHtmlEndpoint);
    if ($htmlContent === false) {
        die("HTML çekilemedi.");
    }

    echo $htmlContent;
}

/**
 * Yaklaşan etkinlikleri API'den çeker ve gösterir
 */
function showUpcomingMatchesFromApi() {
    global $bet_node;
    $lastFetchedUrl = $bet_node . "/upcoming-lastfetched";
    $matchesEndpoint = $bet_node . "/upcoming-matches";
    $matchesHtmlEndpoint = $bet_node . "/upcoming-matches.html";

    // lastfetched zamanını al
    $lastFetchedData = file_get_contents($lastFetchedUrl);
    $lastFetched = null;

    if ($lastFetchedData !== false) {
        $json = json_decode($lastFetchedData, true);
        if (isset($json['lastFetched'])) {
            $lastFetched = strtotime($json['lastFetched']);
        }
    }

    $oneHourAgo = time() - 600;

    // 1 saatten eskiyse /upcoming-matches endpoint'ini çağır ve veri güncelle
    if ($lastFetched === null || $lastFetched < $oneHourAgo) {
        $update = file_get_contents($matchesEndpoint);
        if ($update === false) {
            die("Yaklaşan etkinlik verisi çekilemedi.");
        }
    }

    // API'den güncel HTML'i çek ve yansıt
    $htmlContent = file_get_contents($matchesHtmlEndpoint);
    if ($htmlContent === false) {
        die("Yaklaşan etkinlik HTML'i çekilemedi.");
    }

    echo $htmlContent;
}

/**
 * Game URL'lerini döndürür
 * @param string $gameId Oyun ID'si
 * @return string Oyun URL'si
 */
function getGameUrl($gameId = 'sweet-bonanza-pragmatic') {
    global $bet_node, $available_games;
    
    // Önce Pragmatic Play oyunu olup olmadığını kontrol et
    require_once __DIR__ . '/direct_games.php';
    if (isPragmaticPlayGame($gameId) && hasDirectGameUrl($gameId)) {
        return getDirectGameUrl($gameId);
    }
    
    // Game ID'ye göre URL'yi belirle (proxy üzerinden)
    $gamePath = isset($available_games[$gameId]) ? $available_games[$gameId] : $available_games['sweet-bonanza-pragmatic'];
    
    return $bet_node . $gamePath;
}

/**
 * Mevcut game ID'yi session'dan alır
 * @return string Game ID
 */
function getCurrentGameId() {
    return isset($_SESSION['game_id']) ? $_SESSION['game_id'] : 'sweet-bonanza-pragmatic';
}

/**
 * Oyunun direct URL kullanıp kullanmadığını kontrol eder
 * @param string $gameId Oyun ID'si
 * @return bool True eğer direct URL kullanıyorsa
 */
function isDirectGame($gameId) {
    require_once __DIR__ . '/direct_games.php';
    return isPragmaticPlayGame($gameId) && hasDirectGameUrl($gameId);
}

/**
 * Oyunun iframe'de açılıp açılmayacağını kontrol eder
 * @param string $gameId Oyun ID'si
 * @return bool True eğer iframe'de açılacaksa
 */
function shouldOpenInIframe($gameId) {
    return isDirectGame($gameId);
}


?>
